/*******************************************************************************
**
** Filename:  lmsrtefunctions.js
**
** File Description:  This file contains several javascript variable definitions
**                   and functions that are used commonly by all of the SCO HTML
**                   files in the LMS Runtime Environment Conformance Test.  It
**                   is intended to be included in each SCO HTML file.
**
** Author: ADL Technical Team
**
** Contract Number:
**
**
** Design Issues:   None
** Implementation Issues:  None

** Known Problems: None
** Side Effects:  None
**
** References:
**
**
********************************************************************************
**
** The Advanced Distributed Learning Co-Laboratory (ADL Co-Lab) Hub grants you
** ("Licensee") a non- exclusive, royalty free, license to use and redistribute
** this software in source and binary code form, provided that i) this copyright
** notice and license appear on all copies of the software; and ii) Licensee does
** not utilize the software in a manner which is disparaging to ADL Co-Lab Hub. 
**
** This software is provided "AS IS," without a warranty of any kind. ALL EXPRESS
** OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY IMPLIED
** WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-
** INFRINGEMENT, ARE HEREBY EXCLUDED. ADL Co-Lab Hub AND ITS LICENSORS SHALL NOT
** BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
** OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL ADL Co-Lab Hub
** OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
** INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED
** AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
** INABILITY TO USE SOFTWARE, EVEN IF ADL Co-Lab Hub HAS BEEN ADVISED OF THE
** POSSIBILITY OF SUCH DAMAGES. 
**
*******************************************************************************/

// Define exception/error codes
var _NoError = "0";

// page scoped variable definitions
var apiHandle = null;
var findAPITries = 0;
var noAPIFound = false;

var url = window.location.href;
var query1 = "";
var query2 = "";

// this variable stores an value returned by the LMS
var lmsReturned = null;
var launchTestPassed = false;
var functionTestPassed = false;

// Data Model Conformance State values
//var allElements = true;

var _Debug = false;  // set this to false to turn debugging off
                     // to get rid of those annoying alert boxes.

//define the log message type constants
var _INFO       = 0;  //  0 - informational (diagnostic, trace, etc.)
var _WARNING    = 1;  //  1 - warning
var _PASSED     = 2;  //  2 - conformance check passed
var _FAILED     = 3;  //  3 - conformance check failure
var _TERMINATE  = 4;  //  4 - test suite termination due to nonconformance or
                      //      error
var _CONFORMANT = 5;  //  5 - subject is found to be conformant
var _OTHER      = 9;  //  9 - other

// define which log to write to
var _DETAILED_LOG = "Detailed";
var _SUMMARY_LOG  = "Summary";
var _BOTH_LOG     = "Both";


var _TERMINATED = "terminated";

// local variable definitions

// we'll track the status of the test using a state variable called scoStatus
// This is set by each SCO as it progresses through the test.
var scoStatus = null;

/*******************************************************************************
**
** Function: setScoStatus()
** Inputs:  param
** Return:  None
**
**  Description:
**
*******************************************************************************/
function setScoStatus( param )
{
   LMSTestSCODriver.setSCOStatus( param );
}

/*******************************************************************************
**
** Function: getScoStatus()
** Inputs:  None
** Return:  String
**
**  Description:
**
*******************************************************************************/
function getScoStatus()
{
   return LMSTestSCODriver.getSCOStatus();
}

/*******************************************************************************
 **
 ** Function: writeLogEntry(type, msg)
 ** Inputs:  type - must be one of the following (constants defined above:
 **                      _INFO    - informational (diagnostic, trace, etc.)
 **                      _WARNING - warning
 **                      _PASSED  - conformance check passed
 **                      _FAILED  - conformance check failure
 **                      _TERMINATE - terminating due to nonconformance or error
 **                      _CONFORMANT - subject is  conformant
 **                      _OTHER      - display no icon and use default font.
 **          msg - string containing log message
 **
 ** Return:  None
 **
 **  Description: This function displays a test suite log message.  Note: the
 **  LogWriterApplet must be present in the HTML file that this script is
 **  included in and be identified by the logWriter object id.
 **
 *******************************************************************************/
function writeLogEntry( log, type, msg )
{
   LMSTestSCODriver.writeToLog( log, type, msg );
}

/**********************************************************************
**
** Function setParameters()
** Inputs: None
** Return: None
**
** Description:
** Sets the global variables to track the parameters coming in.  We
** will assume that uri fragments will not be used for the parameters.
** We also expect the parameters to be in the format "?co=<num1>&sco=<num2>",
** where <num1> represents the number associated with the content object
** and <num2> represents the number associated with the sco.
**
**********************************************************************/
function setParameters()
{
   var indexOfParameters = url.indexOf("?");
   var parameters;
   var parameter1 = "";
   var parameter2 = "";

   if ( indexOfParameters != -1 )
   {
      parameters = url.substring( indexOfParameters, url.length );

      // there will only be 2 parameter queries
      indexOfParameters = parameters.indexOf("&");
      parameter1 = parameters.substring( 1, indexOfParameters );
      parameter2 = parameters.substring( (indexOfParameters + 1),
                                         parameters.length );
                                         
	  // we want to split parameter1 and set the second part to query 1
	  tempParamHolder = parameter1.split("=");
	  query1 = tempParamHolder[1];
	  
	  // we want to split parameter2 and set the second part to query 2                                        
  	  tempParamHolder = parameter2.split("=");
	  query2 = tempParamHolder[1];
   }
}

/******************************************************************************
 **
 ** Function getAPIHandle()
 ** Inputs:  None
 ** Return:
 **
 ** Description:
 **
 **
 ******************************************************************************/
function testSCOLaunch()
{
   setParameters();
   var result = LMSTestSCODriver.evalID( query1 + "!" + query2 );

   return result;
}

/*******************************************************************************
 **
 ** Function terminateTest()
 ** Inputs:  None
 ** Return:  None
 **
 ** Description:
 ** This function terminates the current test when a non-conformance
 ** condition is encountered.
 **
 ******************************************************************************/
function terminateTest()
{
   LMSTestSCODriver.terminateTest();
}

/*******************************************************************************
**
** This function looks for an object named API in parent and opener windows
**
** Inputs:  Object - The Window Object
**
** Return:  Object - If the API object is found, it's returned, otherwise null
**          is returned
**
*******************************************************************************/
function findAPI( win )
{
   while ( (win.API_1484_11 == null) &&
           (win.parent != null) &&
           (win.parent != win) )
   {
      findAPITries++;

      if ( findAPITries > 500 )
      {
         alert( "Error finding API -- too deeply nested." );
         return null;
      }

      win = win.parent;
   }

   return win.API_1484_11;
}

/*******************************************************************************
**
** This function looks for an object named API, first in the current window's
** frame hierarchy and then, if necessary, in the current window's opener window
** hierarchy (if there is an opener window).
**
** Inputs:  none
**
** Return:  Object - If the API object is found, it's returned, otherwise null
**                   is returned
**
*******************************************************************************/
function getAPI()
{
   var theAPI = findAPI( window );

   if ( (theAPI == null) &&
        (window.opener != null) &&
        (typeof(window.opener) != "undefined") )
   {
      theAPI = findAPI( window.opener );
   }

   if (theAPI == null)
   {
      alert( "Unable to locate the LMS's API Implementation.\n" +
             "Communication with the LMS will not occur." );

      noAPIFound = true;
   }

   return theAPI
}

/*******************************************************************************
**
** Returns the handle to API object if it was previously set, otherwise it
** returns null
**
** Inputs:  None
**
** Return:  Object - The value contained by the apiHandle variable.
**
*******************************************************************************/
function getAPIHandle()
{
   if ( apiHandle == null )
   {
      if ( noAPIFound == false )
      {
         apiHandle = getAPI();
      }
   }
   return apiHandle;
}

/*******************************************************************************
 **
 ** Function handleAPINotFound()
 ** Inputs:  None
 ** Return:  None
 **
 ** Description:
 ** This function is called when the API object is not found, or is null when a
 ** non-null value is expected.  It logs an appropriate error and terminates the
 ** current test.
 **
 *******************************************************************************/
function handleAPINotFound()
{
   writeLogEntry( _BOTH_LOG, _FAILED, "Unable to locate the LMS API object" );
   LMSTestSCODriver.trackAPIConformance( false );
   terminateTest();
}

////////////////////////////////////////////////////////////////////////////////
//////                                                                    //////
//////   Functions used to interact directly with the test subject LMSs   //////
//////   API Functions                                                    //////
//////                                                                    //////
////////////////////////////////////////////////////////////////////////////////

/*******************************************************************************
**
** Function: doInitialize()
** Inputs:  param - parameter to be given when calling the LMSs Initialize()
**                  function.
** Return:  None
**
** Description:
** Initialize communication with LMS by calling the Initialize
** function which will be implemented by the LMS.
**
*******************************************************************************/
function doInitialize( param )
{
   var api = getAPIHandle();

   if ( api == null )
   {
      alert( "Unable to locate the LMS's API Implementation.\nInitialize() " +
             "was not successful." );

      LMSTestSCODriver.trackAPIConformance( false );
      terminateTest();

      return;
   }

   try
   {
      // call Initialize to the LMSs API
      var result = api.Initialize( param );
      lmsReturned = result.toString();
   }
   catch ( e )
   {
      writeLogEntry( _DETAILED_LOG, _FAILED, "Initialize( String ) NOT " +
                     "found." );

      driver.trackAPIConformance(false);
      terminateTest();
      return;
   }
}

/*******************************************************************************
**
** Function: doTerminate()
** Inputs:  param - parameter to be given when calling the LMSs Terminate()
**                  function.
** Return:  None
**
** Description:
** Terminate communication with LMS by calling the Terminate
** function which will be implemented by the LMS.
**
*******************************************************************************/
function doTerminate( param )
{
   var api = getAPIHandle();

   if ( api == null )
   {
      alert( "Unable to locate the LMS's API Implementation.\nTerminate() " +
             "was not successful." );

      LMSTestSCODriver.trackAPIConformance( false );
      terminateTest();

      return;
   }

   try
   {
      // call Terminate to the LMSs API
      var result = api.Terminate( param );
      lmsReturned = result.toString();
   }
   catch ( e )
   {
      writeLogEntry( _DETAILED_LOG, _FAILED, "Terminate( String ) NOT found." );

      driver.trackAPIConformance(false);
      terminateTest();
      return;
   }
}

/*******************************************************************************
**
** Function: doCommit()
** Inputs:  param - parameter to be given when calling the LMSs Commit()
**                  function.
** Return:  None
**
** Description:
** Commit communication with LMS by calling the Commit
** function which will be implemented by the LMS.
**
*******************************************************************************/
function doCommit( param )
{
   var api = getAPIHandle();

   if ( api == null )
   {
      alert( "Unable to locate the LMS's API Implementation.\nCommit() " +
             "was not successful." );

      LMSTestSCODriver.trackAPIConformance( false );
      terminateTest();

      return;
   }

   try
   {
      // call Commit to the LMSs API
      var result = api.Commit( param );
      lmsReturned = result.toString();
   }
   catch ( e )
   {
      writeLogEntry( _DETAILED_LOG, _FAILED, "Commit( String ) NOT found." );

      driver.trackAPIConformance(false);
      terminateTest();
      return;
   }
}

/*******************************************************************************
**
** Function: doGetValue()
** Inputs:  param - parameter to be given when calling the LMSs GetValue()
**                  function.
** Return:  None
**
** Description:
** GetValue communication with LMS by calling the GetValue
** function which will be implemented by the LMS.
**
*******************************************************************************/
function doGetValue( param )
{
   var api = getAPIHandle();

   if ( api == null )
   {
      alert( "Unable to locate the LMS's API Implementation.\nGetValue() " +
             "was not successful." );

      LMSTestSCODriver.trackAPIConformance( false );
      terminateTest();

      return;
   }

   try
   {
      // call GetValue to the LMSs API
      var result = api.GetValue( param );
      lmsReturned = result.toString();
   }
   catch ( e )
   {
      writeLogEntry( _DETAILED_LOG, _FAILED, "GetValue( String ) NOT found." );

      driver.trackAPIConformance(false);
      terminateTest();
      return;
   }
}

/*******************************************************************************
**
** Function: doSetValue()
** Inputs:  param - parameter to be given when calling the LMSs SetValue()
**                  function.
** Return:  None
**
** Description:
** SetValue communication with LMS by calling the SetValue
** function which will be implemented by the LMS.
**
*******************************************************************************/
function doSetValue( param1, param2 )
{
   var api = getAPIHandle();

   if ( api == null )
   {
      alert( "Unable to locate the LMS's API Implementation.\nSetValue() " +
             "was not successful." );

      LMSTestSCODriver.trackAPIConformance( false );
      terminateTest();

      return;
   }

   try
   {// call SetValue to the LMSs API
      var result = api.SetValue( param1, param2 );
      lmsReturned = result.toString();
   }
   catch ( e )
   {
      writeLogEntry( _DETAILED_LOG, _FAILED, "SetValue( String, String ) " +
                     "NOT found." );

      driver.trackAPIConformance(false);
      terminateTest();
      return;
   }
}

/*******************************************************************************
**
** Function: doGetLastError()
** Inputs:  param - parameter to be given when calling the LMSs GetLastError()
**                  function.
** Return:  None
**
** Description:
** GetLastError communication with LMS by calling the GetLastError
** function which will be implemented by the LMS.
**
*******************************************************************************/
function doGetLastError()
{
   var api = getAPIHandle();

   if ( api == null )
   {
      alert( "Unable to locate the LMS's API Implementation.\nGetLastError() " +
             "was not successful." );

      LMSTestSCODriver.trackAPIConformance( false );
      terminateTest();

      return;
   }

   try
   {
      // call GetLastError to the LMSs API
      var result = api.GetLastError();
      lmsReturned = result.toString();
   }
   catch ( e )
   {
      writeLogEntry( _DETAILED_LOG, _FAILED, "GetLastError() NOT found." );

      driver.trackAPIConformance(false);
      terminateTest();
      return;
   }
}

/*******************************************************************************
**
** Function: doGetErrorString()
** Inputs:  param - parameter to be given when calling the LMSs GetErrorString()
**                  function.
** Return:  None
**
** Description:
** GetErrorString communication with LMS by calling the GetErrorString
** function which will be implemented by the LMS.
**
*******************************************************************************/
function doGetErrorString( param )
{
   var api = getAPIHandle();

   if ( api == null )
   {
      alert( "Unable to locate the LMS's API Implementation.\nGetErrorString() " +
             "was not successful." );

      LMSTestSCODriver.trackAPIConformance( false );
      terminateTest();

      return;
   }

   try
   {
      // call GetErrorString to the LMSs API
      var result = api.GetErrorString( param );
      lmsReturned = result.toString();
   }
   catch ( e )
   {
      writeLogEntry( _DETAILED_LOG, _FAILED, "GetErrorString( String ) " +
                     "NOT found." );

      driver.trackAPIConformance(false);
      terminateTest();
      return;
   }
}

/*******************************************************************************
**
** Function: doGetDiagnostic()
** Inputs:  param - parameter to be given when calling the LMSs GetDiagnostic()
**                  function.
** Return:  None
**
** Description:
** GetDiagnostic communication with LMS by calling the GetDiagnostic
** function which will be implemented by the LMS.
**
*******************************************************************************/
function doGetDiagnostic( param )
{
   var api = getAPIHandle();

   if ( api == null )
   {
      alert( "Unable to locate the LMS's API Implementation.\nGetDiagnostic() " +
             "was not successful." );

      LMSTestSCODriver.trackAPIConformance( false );
      terminateTest();

      return;
   }

   try
   {
      // call GetDiagnostic to the LMSs API
      var result = api.GetDiagnostic( param );
      lmsReturned = result.toString();
   }
   catch ( e )
   {
      writeLogEntry( _DETAILED_LOG, _FAILED, "GetDiagnostic( String ) " +
                     "NOT found." );

      driver.trackAPIConformance(false);
      terminateTest();
      return;
   }
}

function callTerminate()
{
   // test for a successful call to Terminate()
   writeLogEntry( _DETAILED_LOG, _INFO,
                  "Attempting to call Terminate(&quot;&quot;)" );

   var result = driver.testTerminate( "" ) && result;
}

function printDate()
{
   var currentDate = new Date( );

   writeLogEntry( _BOTH_LOG, _INFO, "@@@  " );
   writeLogEntry( _BOTH_LOG, _INFO, "@@@  time = " + currentDate.toLocaleString() );
   writeLogEntry( _BOTH_LOG, _INFO, "@@@  " );
}
